﻿using gov.va.med.vbecs.BOL;
using System;
using System.Linq;
using System.Collections.Generic;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.GUI;
using System.Windows.Forms;

namespace WPF.PresentationLayer.UC115.ViewModels
{
    public class SpecimenXMatchItemViewModel : SpecimenItemViewModel
    {
        private VbecsBaseForm _vbecsBaseForm;

        /// <summary>
        /// PendingSpecimenTestModel
        /// </summary>
        public PendingSpecimenXMatchModel PendingSpecimenXMatchModel
        {
            get
            {
                return PendingTestModel as PendingSpecimenXMatchModel;
            }
            set
            {
                PendingTestModel = value;
                RaisePropertyChanged(() => PendingSpecimenXMatchModel);
            }
        }

        /// <summary>
        /// ProductCode setup as a property here so we can RaisePropertyChanged
        /// which will trigger an update in the UI
        /// </summary>
        private BloodUnitModel _bloodUnitModel;
        public BloodUnitModel BloodUnitModel
        {
            get
            {
                return _bloodUnitModel;
            }
            set
            {
                _bloodUnitModel = value;
                RaisePropertyChanged(() => BloodUnitModel);
                // Re-Validate the test when the Blood Unit changes
                TestValidation = PendingSpecimenXMatchModel.Validate(_bloodUnitModel);
                RaisePropertyChanged(() => EnableXMatchSelection);
            }
        }

        private IList<XMatchInterpOption> _xMatchInterpList;
        public IList<XMatchInterpOption> XMatchInterpList
        {
            get
            {
                return _xMatchInterpList;
            }
            set
            {
                _xMatchInterpList = value;
                RaisePropertyChanged(() => XMatchInterpList);
            }
        }

        private XMatchInterpOption _previousSelectedXMatchInterp;
        private XMatchInterpOption _selectedXMatchInterp;
        public XMatchInterpOption SelectedXMatchInterp
        {
            get
            {
                return _selectedXMatchInterp;
            }
            set
            {
                _previousSelectedXMatchInterp = _selectedXMatchInterp;
                _selectedXMatchInterp = value;
                RaisePropertyChanged(() => SelectedXMatchInterp);

                if (!DisplayOverrideExceptionIfNeeded(_selectedXMatchInterp))
                {
                    System.Windows.Application.Current.Dispatcher.BeginInvoke(new System.Action(() =>
                    {
                        SelectedXMatchInterp = _previousSelectedXMatchInterp;
                    }));
                }              
            }
        }
        public bool EnableXMatchSelection
        {
            get
            {
                return BloodUnitModel != null &&
                       TestValidation != null &&
                       TestValidation.TestValidationStatus != TestValidationStatus.Error;
            }
        }

        public string ISInterpText
        {
            get
            {
                return GetInterpDisplayText(TestType.IS_Interp, PendingSpecimenXMatchModel.ISResult);
            }
        }

        public string AHGInterpText
        {
            get
            {
                return GetInterpDisplayText(TestType.AHG_Interp, PendingSpecimenXMatchModel.AHGResult);
            }
        }

        /// <summary>
        /// PromptForBloodUnit
        /// </summary>
        public bool PromptForBloodUnit { get; private set; }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="pendingSpecimenTestModel"></param>
        public SpecimenXMatchItemViewModel(IPendingSpecimenTestModel pendingSpecimenTestModel, VbecsBaseForm vbecsBaseForm)
            : base(pendingSpecimenTestModel)
        {
            if (pendingSpecimenTestModel == null) throw new ArgumentNullException("pendingSpecimenTestModel");

            _vbecsBaseForm = vbecsBaseForm;
            var pendingSpecimenXMatchModel = pendingSpecimenTestModel as PendingSpecimenXMatchModel;

            if (pendingSpecimenXMatchModel != null && pendingSpecimenTestModel != null)
            {
                PromptForBloodUnit = pendingSpecimenXMatchModel.BloodUnitModel == null;

                // Make new references via clone so we can check for unsaved changes easier
                if (pendingSpecimenXMatchModel.BloodUnitModel != null)
                {
                    BloodUnitModel = pendingSpecimenXMatchModel.BloodUnitModel.Clone();
                }

                // BR_115.29
                XMatchInterpList = pendingSpecimenXMatchModel.GetXMatchInterpOptions();

                if (XMatchInterpList != null &&
                    pendingSpecimenXMatchModel.XMInterp != null)
                {
                    SelectedXMatchInterp = XMatchInterpList.FirstOrDefault(x => x.TestResultId == pendingSpecimenXMatchModel.XMInterp.TestResultId);
                }
            }
        }

        /// <summary>
        /// Check for unsaved changes
        /// </summary>
        public override bool IsDirty()
        {
            return (ReviewComment != PendingTestModel.ReviewComment)
                || !Equals(BloodUnitModel, PendingSpecimenXMatchModel.BloodUnitModel)
                || !Equals(TestValidation, PendingSpecimenXMatchModel.TestValidation);
        }

        private string GetInterpDisplayText(TestType testType, PendingTestResultModel testResult)
        {
            var interp = PendingSpecimenXMatchModel.GetPendingTestResult(testType, true);
            int testResultId;

            if (interp == null &&
                testResult != null &&
                int.TryParse(testResult.TestResultId, out testResultId))
            {
                if (testResultId > 0)
                {
                    return "Incompatible";
                }
                else
                {
                    return "Compatible";
                }
            }
            else if (interp != null)
            {
                if (interp.TestResultId == PendingTestHelper.INCOMPATIBLE ||
                    interp.TestResultId == PendingTestHelper.INCOMPATIBLE_ONLY_MEDICAL_DIRECTOR_ID)
                {
                    return "Incompatible";
                }
                else if (interp.TestResultId == PendingTestHelper.COMPATIBLE_DONT_TRANSFUSE_ID ||
                         interp.TestResultId == PendingTestHelper.COMPATIBLE_SAFE_TO_TRANSFUSE_ID)
                {
                    return "Compatible";
                }
            }

            return PendingSpecimenXMatchModel.GetPendingTestResult(testType, false).TestResultText;
        }

        /// <summary>
        /// Collect an override exception if the user selects Incompatible - Give with Medical Director approval
        /// </summary>
        /// <param name="_selectedXMatchInterp"></param>
        private bool DisplayOverrideExceptionIfNeeded(XMatchInterpOption selectedXMatchInterp)
        {
            if (TestValidation == null)
            {
                return false;
            }
            if (selectedXMatchInterp != null &&
                selectedXMatchInterp.TestResultId == PendingTestHelper.INCOMPATIBLE_ONLY_MEDICAL_DIRECTOR_ID)
            {
                var dlgMsg = "Crossmatch Incompatible: Give Only with Medical Director Approval";
                var exceptionForDisplay = new ExceptionForDisplayModel(ExceptionType.Crossmatch_incompatible_Give_only_with_MD_approval, dlgMsg, dlgMsg, "XM");

                // User has already entered an override for this exception type
                DlgOverrideComment dlg = new DlgOverrideComment(exceptionForDisplay.PromptCaption,
                                                                exceptionForDisplay.PromptMessage,
                                                                exceptionForDisplay.CannedCommentCategory);

                // Show the dialog
                var dlgResult = dlg.ShowDialog(_vbecsBaseForm);

                if (dlgResult == DialogResult.OK)
                {
                    if (TestValidation.ExceptionForDisplayList == null)
                    {
                        TestValidation.ExceptionForDisplayList = new List<ExceptionForDisplayModel>();
                    }

                    exceptionForDisplay.OverrideMessage = dlg.Comment;
                    TestValidation.ExceptionForDisplayList.Add(exceptionForDisplay);
                }
                else
                {
                    return false;
                }
            }
            else if (TestValidation.ExceptionForDisplayList != null)
            {
                // Remove override exception from exception list
                var exception = TestValidation.ExceptionForDisplayList.FirstOrDefault(x => x.ExceptionType == ExceptionType.Crossmatch_incompatible_Give_only_with_MD_approval);
                if (exception != null)
                {
                    TestValidation.ExceptionForDisplayList.Remove(exception);
                }
            }

            return true;
        }
    }
}
